Framework openb3d.B3dglgraphics
Import brl.timerdefault
Import brl.random
Import brl.eventqueue
Import brl.FreeTypeFont
Import bah.bass
Import maxgui.drivers

' user vars
Local equalizer:String = CurrentDir() + "/shader/led1"
Local songtitle:String = CurrentDir() + "/music/2raumwohnung - Wir trafen uns in einem Garten"
Local songext:String = "mp3"

Local gw:Int = 1920
Local gh:Int = 1080
Local sw:Int = 400
Local sh:Int = 240
Local peaks:Int = 2048
Local res:Int = 512'peaks / 4

' option flags
Global OPTION_WAVEFORM:Int = True
Global OPTION_VISUALIZER:Int = True
Global OPTION_FREQUENCY:Int = True
Global OPTION_FULLSCREEN:Int = False

' frequency analyzer
Local frequencies:Float[] = [32.7, 36.7, 41.2, 43.7, 49.0, 55.0, 61.7, 65.4, 73.4, 82.4, 87.3, 98.0, 110.0, 123.5, 130.8, 146.8, 164.8, 174.6, 196.0, 220.0, 246.9, 261.6, 293.7, 329.6, 349.2, 392.0, 440.0, 493.9, 523.3, 587.3, 659.3, 698.5, 784.0, 880.0, 987.8, 1046.5, 1174.7, 1318.5, 1396.9, 1568.0, 1760.0, 1975.5]
Local notecount:Float[frequencies.Length]
Local lightintensity:Int[frequencies.Length]
Local sharpness:Float = 2.0
		
' other vars
Local fps:Int = DesktopHertz()
Local posSize:Long
Local spectrum:TBank = CreateBank(peaks * 4)
Local _left:Int
Local _right:Int

' frequency analysis
Local lows:Float
Local mids:Float
Local high:Float
Local lmax:Float
Local mmax:Float
Local hmax:Float

Local fmx:Float = 0
Local fmn:Float = 0

' timing
Local time:Float
Local timer:TTimer = CreateTimer(fps)

' MAXGUI
Global MAXGUI_Resolutions:String[999]
Global MAXGUI_CURRENT_Width:Int
Global MAXGUI_CURRENT_Height:Int

' initialize window
Local MAXGUI_WINDOW:TGadget = CreateWindow("Blitzmax Shader Equalizer", ClientWidth(Desktop()) / 2 - ((sw) / 2), ClientHeight(Desktop()) / 2 - ((sh) / 2), sw, sh, Null, WINDOW_TITLEBAR | WINDOW_CLIENTCOORDS)

' encapsulating panel
Local MAXGUI_PANEL:TGadget = CreatePanel(10, 10, sw - 20, sh - 20, MAXGUI_WINDOW, PANEL_GROUP, "Select Shader/Music")

' first file selector
CreateLabel("Fragment Shader:", 10, 10, sw, 20, MAXGUI_PANEL)
Local MAXGUI_INPUT_SHADER:TGadget = CreateTextField(10, 30, sw - 140, 20, MAXGUI_PANEL)
Local MAXGUI_INPUT_SHADER_BUTTON:TGadget = CreateButton("Select File", sw - 120, 30, 80, 20, MAXGUI_PANEL)
SetGadgetText(MAXGUI_INPUT_SHADER, Replace(equalizer, CurrentDir(), ""))

' second file selector
CreateLabel("Music file:", 10, 60, sw, 20, MAXGUI_PANEL)
Local MAXGUI_INPUT_MUSIC:TGadget = CreateTextField(10, 80, sw - 140, 20, MAXGUI_PANEL)
Local MAXGUI_INPUT_MUSIC_BUTTON:TGadget = CreateButton("Select File", sw - 120, 80, 80, 20, MAXGUI_PANEL)
SetGadgetText(MAXGUI_INPUT_MUSIC, Replace(songtitle, CurrentDir(), ""))

' options
Local MAXGUI_OPTION_WAVEFORM:TGadget = CreateButton(" Show Waveform", 10, sh - 125, 120, 20, MAXGUI_PANEL, BUTTON_CHECKBOX)
Local MAXGUI_OPTION_VISUALIZER:TGadget = CreateButton(" Show Visualizer", 10, sh - 105, 120, 20, MAXGUI_PANEL, BUTTON_CHECKBOX)
Local MAXGUI_OPTION_FREQUENCY:TGadget = CreateButton(" Show Analyzer", 10, sh - 85, 120, 20, MAXGUI_PANEL, BUTTON_CHECKBOX)
Local MAXGUI_OPTION_FULLSCREEN:TGadget = CreateButton(" Fullscreen", 10, sh - 65, 120, 20, MAXGUI_PANEL, BUTTON_CHECKBOX)
SetButtonState(MAXGUI_OPTION_WAVEFORM, OPTION_WAVEFORM)
SetButtonState(MAXGUI_OPTION_VISUALIZER, OPTION_VISUALIZER)
SetButtonState(MAXGUI_OPTION_FREQUENCY, OPTION_FREQUENCY)
SetButtonState(MAXGUI_OPTION_FULLSCREEN, OPTION_FULLSCREEN)

' resolution
Global MAXGUI_Graphics_Resolution:TGadget = CreateComboBox(140, sh - 125, 220, 32, MAXGUI_PANEL)
MAXGUI_GetResolutions(gw, gh)

' start button
Local MAXGUI_BUTTON_START:TGadget = CreateButton("Start", sw - 120, sh - 80, 80, 30, MAXGUI_PANEL)
ActivateGadget(MAXGUI_BUTTON_START)

Local r1:String
Local r2:Int

' loop until user action
Repeat

	' Wait for an Event
	Select WaitEvent()

		' Window has been closed
		Case EVENT_WINDOWCLOSE, EVENT_APPTERMINATE
		
			End
		
		Case EVENT_GADGETACTION
		
			Select EventSource()

				Case MAXGUI_BUTTON_START
				
					' retrieve selected resolution
					r1 = MAXGUI_Resolutions[SelectedGadgetItem(MAXGUI_Graphics_Resolution)]
					r2 = Instr(r1, "x", 1)

					gw = Int(Mid(r1, 0, r2))
					gh = Int(Mid(r1, r2 + 1, Len(r1)))
				
					Exit
					
				Case MAXGUI_INPUT_SHADER_BUTTON
					Local file:String = RequestFile("Select Equalizer Fragment Shader", "Fragment Shaders:frag;frag", False, CurrentDir() + "/shader/")
					If file Then
						equalizer = StripExt(file)
						SetGadgetText(MAXGUI_INPUT_SHADER, Replace(equalizer, CurrentDir(), ""))
					EndIf
					
				Case MAXGUI_INPUT_MUSIC_BUTTON
					Local file:String = RequestFile("Select Music File to visualize", "mp3,mod,flac,wav", False, CurrentDir() + "/music/")
					If file Then
						songtitle = StripExt(file)
						songext = Lower(ExtractExt(file))
						SetGadgetText(MAXGUI_INPUT_MUSIC, Replace(songtitle, CurrentDir(), ""))
						
						Print songtitle + "|" + songext
						
					EndIf
										
			End Select
			
	End Select

Forever

Print r1 + "|" + r2
Print gw + "|" + gh

' hide panel
HideGadget(MAXGUI_PANEL)

' transfer user settings to variables
OPTION_WAVEFORM = ButtonState(MAXGUI_OPTION_WAVEFORM)
OPTION_VISUALIZER = ButtonState(MAXGUI_OPTION_VISUALIZER)
OPTION_FREQUENCY = ButtonState(MAXGUI_OPTION_FREQUENCY)
OPTION_FULLSCREEN = ButtonState(MAXGUI_OPTION_FULLSCREEN)

' resize MAXGUI window
Local modus:Int = 1
Local canvas:Int = False
If Not OPTION_FULLSCREEN Then

	SetGadgetShape(MAXGUI_WINDOW, ClientWidth(Desktop()) / 2 - ((gw) / 2), ClientHeight(Desktop()) / 2 - ((gh) / 2), gw, gh)
	Local MAXGUI_CANVAS:TGadget = CreateCanvas(0, 0, gw, gh, MAXGUI_WINDOW, 0)
	SetGadgetLayout(MAXGUI_CANVAS, 1, 1, 1, 1)
	ActivateGadget(MAXGUI_CANVAS)
	SetGraphics CanvasGraphics(MAXGUI_CANVAS)
	modus = 2
	canvas = True
	
EndIf

Graphics3D(gw, gh, 0, modus, fps, -1, canvas)
EnablePolledInput()


Print equalizer
Print songtitle + "." + songext


' initialize bass
If Not TBass.Init(-1, 44100, 0, Null, Null) Then DebugLog "Can't initialize device : " + TBass.ErrorGetCode() ; End

Local stream:TBassChannel = New TBassStream.CreateFile(songtitle + "." + songext, 0, 0, 0)

If Not stream Then stream = New TBassMusic.FileLoad(songtitle + "." + songext, 0, 0, BASS_SAMPLE_LOOP | BASS_MUSIC_RAMPS | BASS_MUSIC_PRESCAN, 0)

If stream Then

	posSize = stream.GetLength(BASS_POS_BYTE) / 10.0
	If Not stream.Play(False) Then DebugLog "can't play... : " + TBass.ErrorGetCode()

EndIf

' camera
Local Camera:TCamera = CreateCamera()
CameraRange Camera, 0.1, gw * 2

' fullscreen shader sprite
Local screen:TSprite = CreateSprite(camera)
ScaleSprite screen, gw, gh
PositionEntity screen, 0, 0, gh

' texture and pixmap for equalizer
Local tex:TTexture = CreateTexture(res, 2)
Local pixmap:TPixmap = CreatePixmap(res, 2, PF_RGBA8888)

' shader
Local shader:TShader = LoadShader("", "shader/base.vert", equalizer + ".frag")
SetFloat2(shader, "resolution", gw, gh)
UseFloat(shader, "time", time)
ShaderTexture(shader, tex, "iChannel0", 0)
ShadeEntity(screen, shader)

' main loop
While Not AppTerminate()

	Select WaitEvent()

		' Window has been closed
		Case EVENT_WINDOWCLOSE, EVENT_APPTERMINATE
		
			End
			
	End Select

	If KeyHit(KEY_F1) Then OPTION_WAVEFORM = 1 - OPTION_WAVEFORM
	If KeyHit(KEY_F2) Then OPTION_FREQUENCY = 1 - OPTION_FREQUENCY
	If KeyHit(KEY_F3) Then OPTION_VISUALIZER = 1 - OPTION_VISUALIZER
	
	If KeyHit(KEY_ESCAPE) Then End
	
	If KeyDown(KEY_LEFT) Or KeyDown(KEY_RIGHT) Then
	
		Local k:Int = KeyDown(KEY_RIGHT) - KeyDown(KEY_LEFT)
		stream.SetPosition((stream.GetPosition(BASS_POS_BYTE) + (k * (posSize / 50)) Mod posSize), BASS_POS_BYTE)
		stream.Play(False)
		Delay 10
	
	EndIf
	
	For Local k:Int = 1 To 11
	
		If KeyHit(KEY_0 + (k Mod 10)) Then stream.SetPosition(Int((k - 1 + 0.25) * posSize), BASS_POS_BYTE)
		
	Next
	
	' shader timing
	time = Float((TimerTicks(timer) * 1.0 / fps) * 1.0)
	
	RenderWorld
	
	BeginMax2D()
	
		SetBlend ALPHABLEND

		' get FFT spectrum and levels
		stream.GetData(spectrum.Buf(), BASS_DATA_FFT4096 | BASS_DATA_FLOAT | BASS_DATA_FFT_NOWINDOW)
		stream.GetLevel(_left, _right)
		
		' detailed note frequency array, from C1 to B7, only full tones: C,D,E,F,G,A,B = 6x7 = 42 notes
		' C1 = 32.7   = 1st octave
		' C2 = 65.4   = 2nd octave
		' C3 = 130.8  = 3rd octave
		' C4 = 261.6  = 4th octave (at 440Hz: Cocert pitch)
		' C5 = 523.3  = 5th octave
		' C6 = 1046.5 = 6th octave
		
		fmx = 0
		fmn = 0
		
		' frequency analysis
		For Local i:Int = 0 To peaks Step 4
					
			Local h0:Float = PeekFloat(spectrum, i)
			Local h:Float = Float(Log(h0 + 1) * Exp(h0 + 1 + (i / peaks)))
			
			h = Limit(h, 0, 1)
			If i < 75 Then lows:+h                 ' Low frequencies
			If i >= 75 And i < 250 Then mids:+h    ' Mid frequencies
			If i >= 250 Then high:+h               ' High frequencies
			
			' rough delimitation of the note frequencies
			For Local j:Int = 1 To frequencies.Length - 2
			
				' distance between current and previous frequency, take 1/4th
				Local d1:Float = (frequencies[j] - frequencies[j - 1]) / sharpness
				
				' distance between current and next frequency, take 1/4th
				Local d2:Float = (frequencies[j + 1] - frequencies[j]) / sharpness
			
				' check if distance lies between current note and the two distances, and sum up the value
				If i > frequencies[j - 1] + d1 And i < frequencies[j + 1] - d2 Then notecount[j]:+h
				
				If notecount[j] > fmx Then fmx = notecount[j]
				If notecount[j] < fmn Then fmn = notecount[j]
				
			Next

		Next
		
		' visualize the "caught" note frequencies with blocks
		For Local j:Int = 0 To frequencies.Length - 1
		
			' normalize the quadratic notecount sum to RGB intensity
			' 0...32 ^ 2 gives a range of 0...1024 with clipping at 255
			' so only the intensest notes get a full white bar, the others grey tones
			Local c:Int = Limit(Float(Normalize(notecount[j], fmn, fmx, 0, 32) ^ 2), 0, 255)
			
			' holds the light intensity of the note for later use (ex. shader, external light)
			lightintensity[j] = c
			
			SetBlend SOLIDBLEND

			If OPTION_FREQUENCY Then

				' draw piano
				Local sx:Int = gw / 2 - (frequencies.Length - 2) * 30 / 2
				SetColor c, c, c
				DrawRect(sx + (j * 30), gh * 0.3, 29, 29 * 3)
				
				' draw octave borders
				If (j Mod 7 = 0) Then
				
					SetColor 255, 0, 0
					DrawRect(sx + (j * 30), gh * 0.3, 2, 29 * 3)
					
				EndIf
				
			EndIf
			
			notecount[j]:*0.9
			
		Next

		' decrease, normalize and clip frequency sums
		lows:*0.8 ; If lows > lmax Then lmax = lows
		mids:*0.8 ; If mids > mmax Then mmax = mids
		high:*0.8 ; If high > hmax Then hmax = high
		lows = Limit(Normalize(lows, 0, lmax, 0, 1), 0, 1)
		mids = Limit(Normalize(mids, 0, mmax, 0, 1), 0, 1)
		high = Limit(Normalize(high, 0, hmax, 0, 1), 0, 1)
		
		' convert FFT data to spectrum analyzer pixmap / texture
		For Local i:Int = 0 To res - 1
		
			Local h0:Float = PeekFloat(spectrum, i * (peaks / res))
			Local h:Float = Float(Log(h0 + 1) * Exp(h0 + 1 + (i / peaks)))
			
			Local c:Int = FFTFrequency2Index(h * 65536, peaks, 44100)
			
			c = Limit(c, 0, 255)
			
			Local c1:Int = (_left + _right) * 0.5 * h0 / 16    ' variant1: loadness * FFT
			Local c2:Int = Int(h * 1024)                       ' variant2: logarithmic
			
			' for shader "test.frag"
			'c = h * 65536
			'c1 = h * 65536
			
			If OPTION_WAVEFORM Then
			
				' draw samples
				SetColor c, c, c
				DrawLine(i, 0, i, 32)
				
				' draw FFT (c1 or c2 value!)
				SetColor c1, c1, c1
				DrawLine(i, 32, i, 64)
				
				' red divider line
				SetColor(255, 0, 0)
				DrawLine(0, 32, res - 1, 32)

			EndIf
			
			' write data to shader texture
			WritePixel(pixmap, i, 0, c Shl 16)
			WritePixel(pixmap, i, 1, c2 Shl 16)

		Next
		
		If OPTION_VISUALIZER Then
				
			' draw frequency dots
			DrawDot(Int(gw / 2 - 150 - (100 * (0.5 + lows) / 2)), Int(gh / 2 - (100 * (0.5 + lows) / 2)), 100, 100, 0.5 + lows, 255, 0, 0)
			DrawDot(Int(gw / 2 - (100 * (0.5 + mids) / 2)), Int(gh / 2 - (100 * (0.5 + mids) / 2)), 100, 100, 0.5 + mids, 0, 255, 0)
			DrawDot(Int(gw / 2 + 150 - (100 * (0.5 + high) / 2)), Int(gh / 2 - (100 * (0.5 + high) / 2)), 100, 100, 0.5 + high, 0, 128, 255)
			
			' draw channel levels
			SetColor(255, 255, 255)
			SetAlpha((lows + mids + high / 3))
			SetScale(1.0, 1.0)
			Local lwidth:Int = _left / 150.0
			DrawRect gw / 2 - lwidth, gh * 0.6, lwidth, 20
			DrawRect gw / 2, gh * 0.6, _right / 150.0, 20
			
		EndIf
		
	EndMax2D()

	' copy spectrum analyzer pixmap to spectrum analyzter shader texture buffer
	BufferToTex tex, PixmapPixelPtr(pixmap, 0, 0)

	Flip
		
Wend

End

' limit a value to a given range
Function Limit:Float(v:Float, mn:Float, mx:Float)

	If v > mx Then v = mx Else If v < mn Then v = mn
	Return v

End Function

' draw a big dot
Function DrawDot(x:Int, y:Int, w:Int, h:Int, s:Float, r:Int, g:Int, b:Int)

	SetAlpha s
	SetScale s, s
	SetColor r, g, b
	DrawOval x, y, w, h

End Function

' normalize a value from a given range to a given range
Function Normalize:Float(v:Float, vmin:Float, vmax:Float, nmin:Float, nmax:Float)

	Return ((v - vmin) / (vmax - vmin)) * (nmax - nmin) + nmin
	
End Function

' FFT sample data at a given index is the amplitude (as a float value between 0.0 and 1.0
' where 0.0 means -unlimited dB (silence) and 1.0 means 0 dB (max)) at this frequency.
' 
' Example: at 44100Hz, slot 191 of a 512 sample FFT (191*44100/512) will represent 16451Hz
'
Function FFTFrequency2Index:Int(frequency:Float, fftlength:Int, samplerate:Int)

	Local i:Int = fftlength * frequency / samplerate
	If i > fftlength / 2 - 1 Then i = fftlength / 2 - 1
	Return i

End Function

' ------------------------------------------------------------------------------------------------
' Retrieve and filter available Graphics Resolutions
' ------------------------------------------------------------------------------------------------
Function MAXGUI_GetResolutions(setw:Int = Null, seth:Int = Null)

	Local w:Int
	Local h:Int
	Local d:Int
	Local z:Int

	Local m:Int
	Local i:Int
	
	Local add:Int
	Local index:Int = 0
	
	If (Not setw) Or (Not seth) Then
	
		setw = DesktopWidth()
		seth = DesktopHeight()
			
	EndIf
	
	w = setw
	h = seth
	
	' iterate through all graphic modes
	For m = 0 To CountGraphicsModes() - 1
	
		' get available graphic modes and store them in variables
		GetGraphicsMode(m, w, h, d, z)
		
		' assuming we're adding them
		add = True
		
		' Resolution is already known or height too small? sort them out!
		For i = 0 To index
		
			If z < 60 Then Add = False ; Exit
		
			If MAXGUI_Resolutions[i] = (w + "x" + h) Or h < 720 Then Add = False ; Exit
		
		Next
				
		' add all other resolutions to our Pulldown menu
		If add Then
		
			' add to Gadget
			AddGadgetItem(MAXGUI_Graphics_Resolution, w + "x" + h + " (" + d + "bit, " + z + "Hz)")
			
			' if the current Resolution matches Desktop resolution: preselect it!
			If w = setw And h = seth Then
			
				SelectGadgetItem(MAXGUI_Graphics_Resolution, index)
				MAXGUI_CURRENT_Width = w
				MAXGUI_CURRENT_Height = h
				
			EndIf
			
			' store the new Resolution in Array
			MAXGUI_Resolutions[index] = w + "x" + h
			
			' counter
			index:+1
			
		EndIf
				
	Next

End Function